﻿using Microsoft.Win32;
using System.Collections.ObjectModel;
using System.Windows;
using ThinkGeo.Core;
using ThinkGeo.UI.Wpf;

namespace LegendSample
{
    /// <summary>
    /// Interaction logic for MainWindow.xaml
    /// </summary>
    public partial class MainWindow : Window
    {
        MapPrinterLayer mapPrinterLayer;
        public MainWindow()
        {
            InitializeComponent();
        }

        private async void MapView_Loaded(object sender, RoutedEventArgs e)
        {
            ShapeFileFeatureLayer _crimeLayer;

            mapView.MapUnit = GeographyUnit.Meter;

            var printerZoomLevelSet = new PrinterZoomLevelSet(GeographyUnit.Meter, PrinterHelper.GetPointsPerGeographyUnit(GeographyUnit.Meter));
            mapView.ZoomScales = printerZoomLevelSet.GetScales();
            mapView.MinimumScale = mapView.ZoomScales[mapView.ZoomScales.Count - 1];

            // PrinterInteractiveOverlay allows users to drag/resize print elements on the map
            var printerOverlay = new PrinterInteractiveOverlay();
            // PagePrinterLayer represents the physical paper (e.g., ANSI A / Letter)
            var pageLayer = new PagePrinterLayer(PrinterPageSize.AnsiA, PrinterOrientation.Portrait);

            _crimeLayer = new ShapeFileFeatureLayer(@".\data\Frisco_Crime.shp");
            // Converting from Local State Plane (2276) to Web Mercator (3857)
            _crimeLayer.FeatureSource.ProjectionConverter = new ProjectionConverter(2276, 3857);

            // Using ValueStyle to categorize map points based on the 'OffenseGro' column
            var valueStyle = new ValueStyle("OffenseGro", new Collection<ValueItem>()

            {
                new ValueItem("Burglary", new PointStyle(PointSymbolType.Circle, 12, GeoBrushes.Red) { OutlinePen = new GeoPen(GeoBrushes.DarkBlue, 1)}),
                new ValueItem("Theft", new PointStyle(PointSymbolType.Triangle, 12, GeoBrushes.Blue) { OutlinePen = new GeoPen(GeoBrushes.DarkRed, 1)}),
                new ValueItem("Fraud", new PointStyle(PointSymbolType.Square, 12, GeoBrushes.Blue) { OutlinePen = new GeoPen(GeoBrushes.DarkRed, 1)}),
                new ValueItem("Assault", new PointStyle(PointSymbolType.Cross, 12, GeoBrushes.Blue) { OutlinePen = new GeoPen(GeoBrushes.DarkRed, 1)}),
                new ValueItem("DUI", new PointStyle(PointSymbolType.Diamond,  12, GeoBrushes.Blue) { OutlinePen = new GeoPen(GeoBrushes.DarkRed, 1)}),
                new ValueItem("Weapons", new PointStyle(PointSymbolType.Star, 12, GeoBrushes.Blue) { OutlinePen = new GeoPen(GeoBrushes.DarkRed, 1)}),
                new ValueItem("Robbery", new PointStyle(PointSymbolType.StarCircled, 12, GeoBrushes.Blue) { OutlinePen = new GeoPen(GeoBrushes.DarkRed, 1)}),
                new ValueItem("Harassment", new PointStyle(PointSymbolType.DiamondNarrow, 12, GeoBrushes.Blue) { OutlinePen = new GeoPen(GeoBrushes.DarkRed, 1)}),
                new ValueItem("Hacking", new PointStyle(PointSymbolType.StarCircled, 12, GeoBrushes.Blue) { OutlinePen = new GeoPen(GeoBrushes.DarkRed, 1)}),
            });

            _crimeLayer.ZoomLevelSet.ZoomLevel01.CustomStyles.Add(valueStyle);
            _crimeLayer.ZoomLevelSet.ZoomLevel01.ApplyUntilZoomLevel = ApplyUntilZoomLevel.Level20;
            _crimeLayer.Open();

            // This defines the specific map area that will appear INSIDE the paper boundaries
            mapPrinterLayer = new MapPrinterLayer(new LayerBase[] { _crimeLayer }, _crimeLayer.GetBoundingBox(), GeographyUnit.Meter);

            var pageCenter = pageLayer.GetPosition().GetCenterPoint();
            // Position the map element on the page (Width: 7.5", Height: 5", centered with vertical offset)
            mapPrinterLayer.SetPosition(7.5, 5, pageCenter.X, pageCenter.Y + 1.75, PrintingUnit.Inch);

            printerOverlay.PrinterLayers.Add("pageLayer", pageLayer);
            printerOverlay.PrinterLayers.Add(mapPrinterLayer);

            LegendPrinterLayer legendPrinterLayer = new LegendPrinterLayer();
            legendPrinterLayer.Title = new LegendItem()
            {
                TextStyle = new TextStyle("Crime Categories", new GeoFont("Arial", 12, DrawingFontStyles.Bold), GeoBrushes.Black)
            };

            // Dynamically build legend items based on the ValueStyle defined earlier
            foreach (var velueItem in valueStyle.ValueItems)
            {
                legendPrinterLayer.LegendItems.Add(new LegendItem()
                {
                    // you can customize the style(AreaStyle, LineStyle, and any valid thinkgeo styles) for legend here if needed
                    ImageStyle = velueItem.DefaultPointStyle,
                    TextStyle = new TextStyle(velueItem.Value, new GeoFont("Arial", 10), GeoBrushes.Black),
                });
            }

            // Position legend relative to page center
            legendPrinterLayer.SetPosition(1.5, 2.1, pageCenter.X + 2, pageCenter.Y - 3, PrintingUnit.Inch);
            legendPrinterLayer.ContentResizeMode = LegendContentResizeMode.Resizable;
            legendPrinterLayer.ResizeMode = PrinterResizeMode.Resizable;
            printerOverlay.PrinterLayers.Add(legendPrinterLayer);

            pageLayer.BackgroundMask = AreaStyle.CreateSimpleAreaStyle(GeoColors.White, GeoColors.Black);
            mapView.InteractiveOverlays.Insert(0, "printerOverlay", printerOverlay);
            mapView.CurrentExtent = pageLayer.GetPosition().GetBoundingBox();

            await mapView.RefreshAsync();
        }

        private void Button_Click(object sender, RoutedEventArgs e)
        {
            var printerOverlay = (PrinterInteractiveOverlay)mapView.InteractiveOverlays["printerOverlay"];

            // Use SkiaGeoCanvas for high-quality rendering (cross-platform compatible)
            SkiaGeoCanvas canvas = new SkiaGeoCanvas();
            GeoImage bitmap = new GeoImage((int)mapView.ActualWidth, (int)mapView.ActualHeight);

            canvas.BeginDrawing(bitmap, mapView.CurrentExtent, mapView.MapUnit);

            // Draw layers in reverse (Bottom to Top) to maintain correct Z-order on the export
            foreach (var printerLayer in printerOverlay.PrinterLayers.Reverse())
            {
                printerLayer.IsDrawing = true;
                // We skip drawing the PagePrinterLayer if we only want the content, 
                // though usually drawing it provides the white background.
                if (!(printerLayer is PagePrinterLayer))
                {
                    printerLayer.Draw(canvas, new Collection<SimpleCandidate>());
                }
                printerLayer.IsDrawing = false;
            }
            canvas.EndDrawing();

            SaveFileDialog saveFileDialog = new SaveFileDialog { Filter = "PNG Image (*.png)|*.png", DefaultExt = ".png" };
            if (saveFileDialog.ShowDialog() == true)
            {
                bitmap.Save(saveFileDialog.FileName);
            }
        }
    }
}