using System.IO;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using ThinkGeo.Core;
using ThinkGeo.UI.Wpf;
using System.Text;

namespace Post12316Repro.App;

public partial class MainWindow : Window
{
	private SkiaRasterLayer? skiaRasterLayer;
	private bool mapInitialized;

	public MainWindow()
	{
		InitializeComponent();
		ResamplingModeComboBox.ItemsSource = Enum.GetValues<RasterResamplingMode>();
		ResamplingModeComboBox.SelectedItem = RasterResamplingMode.Auto;
	}

	private async void MapView_Loaded(object sender, RoutedEventArgs e)
	{
		if (mapInitialized)
		{
			return;
		}

		MapView.MapUnit = GeographyUnit.Meter;

		var openStreetMapOverlay = new OpenStreetMapOverlay("Post12316Repro");
		MapView.Overlays.Add(openStreetMapOverlay);

		var rasterFilePath = Path.Combine(AppDomain.CurrentDomain.BaseDirectory, "Data", "input-9x11.png");

		if (!File.Exists(rasterFilePath))
		{
			GenerateInputRasterFile(rasterFilePath);
		}

		var overlay = new LayerOverlay
		{
			TileType = TileType.SingleTile,
		};
		MapView.Overlays.Add(overlay);

		RectangleShape rasterExtent = new RectangleShape(-10_782_500, 3_917_500, -10_773_500, 3_906_500);
		skiaRasterLayer = new SkiaRasterLayer(rasterFilePath, rasterExtent);
		skiaRasterLayer.ResamplingMode = GetSelectedResamplingMode();
		overlay.Layers.Add(skiaRasterLayer);

		MapView.CurrentExtent = rasterExtent;
		var initialScale = MapUtil.GetScale(MapView.MapUnit, rasterExtent, MapView.MapWidth, MapView.MapHeight);
		MapView.CurrentScale = initialScale * 2.8;
		mapInitialized = true;

		await MapView.RefreshAsync();
		WriteDiagnosticLog(openStreetMapOverlay);
	}

	private async void ResamplingModeComboBox_SelectionChanged(object sender, SelectionChangedEventArgs e)
	{
		if (!mapInitialized || skiaRasterLayer == null)
		{
			return;
		}

		skiaRasterLayer.ResamplingMode = GetSelectedResamplingMode();
		await MapView.RefreshAsync();
	}

	private RasterResamplingMode GetSelectedResamplingMode()
	{
		return ResamplingModeComboBox.SelectedItem is RasterResamplingMode mode
			? mode
			: RasterResamplingMode.Auto;
	}
	
	private static void GenerateInputRasterFile(string outputPath)
	{
		Directory.CreateDirectory(Path.GetDirectoryName(outputPath)!);

		const int width = 9;
		const int height = 11;
		var pixels = new byte[width * height * 4];

		var pattern = new[]
		{
			".........",
			".112223..",
			".112223..",
			".11..23..",
			".444556..",
			".444556..",
			".44..56..",
			"..777888.",
			"..777888.",
			"..77..88.",
			"........."
		};

		for (var y = 0; y < height; y++)
		{
			for (var x = 0; x < width; x++)
			{
				var symbol = pattern[y][x];
				var color = GetCellColor(symbol);

				var pixelIndex = ((y * width) + x) * 4;
				pixels[pixelIndex] = color.B;
				pixels[pixelIndex + 1] = color.G;
				pixels[pixelIndex + 2] = color.R;
				pixels[pixelIndex + 3] = color.A;
			}
		}

		var bitmap = BitmapSource.Create(width, height, 96, 96, PixelFormats.Bgra32, null, pixels, width * 4);
		var encoder = new PngBitmapEncoder();
		encoder.Frames.Add(BitmapFrame.Create(bitmap));

		using var stream = new FileStream(outputPath, FileMode.Create, FileAccess.Write, FileShare.None);
		encoder.Save(stream);
	}

	private static (byte R, byte G, byte B, byte A) GetCellColor(char symbol)
	{
		return symbol switch
		{
			'1' => (255, 40, 30, 210),
			'2' => (255, 134, 31, 210),
			'3' => (240, 216, 27, 210),
			'4' => (124, 219, 70, 210),
			'5' => (23, 191, 230, 210),
			'6' => (37, 110, 255, 210),
			'7' => (20, 205, 180, 210),
			'8' => (12, 88, 160, 210),
			_ => (0, 0, 0, 0)
		};
	}

	private void WriteDiagnosticLog(OpenStreetMapOverlay openStreetMapOverlay)
	{
		try
		{
			var extent = MapView.CurrentExtent;
			var center = extent.GetCenterPoint();
			var matrixInfo = openStreetMapOverlay.TileMatrixSet.GetSnappedMatrix(MapView.CurrentScale);
			var matrix = matrixInfo.TileMatrix;
			var zoom = matrixInfo.MatrixIndex;

			var sb = new StringBuilder();
			sb.AppendLine($"Timestamp={DateTime.Now:O}");
			sb.AppendLine($"MapWidth={MapView.MapWidth}");
			sb.AppendLine($"MapHeight={MapView.MapHeight}");
			sb.AppendLine($"MapViewActualWidth={MapView.ActualWidth}");
			sb.AppendLine($"MapViewActualHeight={MapView.ActualHeight}");
			sb.AppendLine($"CurrentScale={MapView.CurrentScale}");
			sb.AppendLine($"CurrentExtent={extent.MinX},{extent.MinY},{extent.MaxX},{extent.MaxY}");
			sb.AppendLine($"Center={center.X},{center.Y}");
			sb.AppendLine($"SnappedZoom={zoom}");
			sb.AppendLine($"SnappedScale={matrix.Scale}");
			sb.AppendLine($"TileWidth={matrix.TileWidth}");
			sb.AppendLine($"TileHeight={matrix.TileHeight}");

			var centerCell = matrix.GetCell(center);
			sb.AppendLine($"CenterTile={centerCell.X},{centerCell.Y}");

			var mapScreenPoint = MapView.PointToScreen(new Point(0, 0));
			sb.AppendLine($"MapViewScreen={mapScreenPoint.X},{mapScreenPoint.Y}");
			var source = PresentationSource.FromVisual(MapView);
			if (source?.CompositionTarget != null)
			{
				var toDevice = source.CompositionTarget.TransformToDevice;
				sb.AppendLine($"DpiScale={toDevice.M11},{toDevice.M22}");
				sb.AppendLine($"MapViewScreenPixels={mapScreenPoint.X * toDevice.M11},{mapScreenPoint.Y * toDevice.M22}");
				sb.AppendLine($"MapViewPixelSize={MapView.ActualWidth * toDevice.M11},{MapView.ActualHeight * toDevice.M22}");
			}

			var path = Path.Combine(AppDomain.CurrentDomain.BaseDirectory, "analysis-diagnostic.txt");
			File.WriteAllText(path, sb.ToString());
		}
		catch
		{
			// best-effort diagnostics only
		}
	}
}
